#! /usr/bin/env Rscript -- vanilla
# install.packages("readxl")
# load required libraries
library(tools)
library(readxl)

# function used to format x-axis as log scale
log10Tck <- function(side, type){
   lim <- switch(side, 
     x = par('usr')[1:2],
     y = par('usr')[3:4],
     stop("side argument must be 'x' or 'y'"))
   at <- floor(lim[1]) : ceiling(lim[2])
   return(switch(type, 
     minor = outer(1:9, 10^(min(at):max(at))),
     major = 10^at,
     stop("type argument must be 'major' or 'minor'")
   ))
}

# determine how to read data based on how R code is being run
if (!interactive()) {
  # R code is being run non-interactively, i.e. by command line or batch file
  args<-commandArgs(trailingOnly = TRUE)
  if (length(args)==0) {
    stop("Need an argument.", call. = FALSE)
  } else {
    plotDataFile<-args[1] # set data file to first argument of command
    savePlot<-TRUE # set savePot to TRUE since we don't ask user
  }
} else {
  # R code is being run interactively, i.e. in RStudio
  plotDataFile<-file.choose() # ask user to specify data file
  tempSel<-menu(c("View","Save"),graphics=TRUE,title="View or Save Plot?") # ask user to specifiy whether to save plot
  savePlot<-(tempSel==2)
}

# set default values for elements if not specified in Excel file
# using c() allows cycling through values for different rows of plot
# for points
def.pch<-c(21)
def.cex<-c(1)
def.bg<-c("white")
def.p.col<-c("black")

# for horizontal lines
def.h.col<-def.p.col
def.h.lty<-c(1)
def.h.lwd<-c(2)

# for vertical lines
def.v.col<-c("black")
def.v.lty<-c(2)
def.v.lwd<-c(1)

# for notes
def.n.cex<-0.7

# Get plot information based on Excel file
# expected column names to check for correct file
colNames<-c("title","...2","label","row...4","font","...6","row...7","type","conc","end","note","pch","cex","bg","col","lty","lwd","comment")

plotDataPath<-dirname(plotDataFile)
plotDataBaseName<-file_path_sans_ext(basename(plotDataFile))
plotDataExt<-file_ext(plotDataFile)

if (plotDataExt=="xlsx" || plotDataExt=="xls") {
  plotTemp<-read_excel(plotDataFile, sheet=2, skip=1) # read worksheet 2 of Excel file skipping first row
  sheetName<-excel_sheets(plotDataFile)[2]
} else {
  if (plotDataExt=="csv") {
    plotTemp<-read.csv(plotDataFile, skip=1, na.strings=c("")) # read csv file skipping first row
    sheetName<-"csv"
  } else {
    stop("Error file extension.", call.=FALSE) # incorrect file extension
  }
}

# simple checks to see if file is in correct format 
if (dim(plotTemp)[2]!=18) { stop("Error # of columns.", call.=FALSE) } # 18 columns
if (any(names(plotTemp)[c(1,3,5,8:18)]!=colNames[c(1,3,5,8:18)])) { stop("Error column headings.", call.=FALSE) } # expected column names
plotTitle<-plotTemp[1,1]
plotXlab<-plotTemp[4,1]
plotLabels<-plotTemp[!is.na(plotTemp$label),3:5]
plotData<-plotTemp[!is.na(plotTemp$type),7:18]
names(plotLabels)[2]<-"row"
names(plotData)[1]<-"row"
# -------------------------------------

if (savePlot) {
  # set up plot for saving to pdf
  plotName<-file.path(plotDataPath,paste0(plotTitle,"_Rplot.pdf")) # set complete path for pdf
  pdf(file=plotName,width=6.5,height=5.5,pointsize=9) # set up pdf including plot size in inches
  par(mar=c(5,15,3,1),cex.axis=1, bty="n")
} else {
  # set up plot for onscreen viewing
  par(mar=c(5,15,3,1),cex.axis=1, bty="n")
}

minRow<-min(c(plotLabels$row,plotData$row), na.rm=TRUE)
maxRow<-max(c(plotLabels$row,plotData$row), na.rm=TRUE)
numRows<-maxRow-minRow
ySpace<-0.4

minX<-min(plotData[,c("conc","end")], na.rm=TRUE)
maxX<-max(plotData[,c("conc","end")], na.rm=TRUE)

plot(c(1,1), c(1,1), type="n", ylim = c(minRow-ySpace, maxRow+ySpace+0.2), xlim = c(minX,maxX), log = "x", xlab="", xaxt="n", yaxt="n", ylab="", bty="n", main=plotTitle, yaxs="i")
title(xlab=plotXlab, line=2)

temp<-plotData[plotData$type=="v",]
if (dim(temp)[1]>0) {
	temp$end[is.na(temp$end)]<-maxRow+ySpace
	temp$col[is.na(temp$col)]<-def.v.col[1]
	temp$lty[is.na(temp$lty)]<-def.v.lty[1]
	temp$lwd[is.na(temp$lwd)]<-def.v.lwd[1]
	arrows(temp$conc, minRow-ySpace, temp$conc, temp$end, lty=temp$lty, col=as.character(temp$col), lwd=temp$lwd, length = 0, angle = 90, code = 3)
	text(temp$conc, temp$end, temp$note, pos=3, cex=def.n.cex)
}

temp<-plotData[plotData$type=="h",]
if (dim(temp)[1]>0) {
	temp$conc[is.na(temp$conc)]<-10^par('usr')[1]
	temp$end[is.na(temp$end)]<-10^par('usr')[2]
	temp$col[is.na(temp$col)]<-def.h.col[1+((floor(temp$row[is.na(temp$col)])-1)%%length(def.h.col))]
	temp$lty[is.na(temp$lty)]<-def.h.lty[1]
	temp$lwd[is.na(temp$lwd)]<-def.h.lwd[1]
	arrows(temp$conc, temp$row, temp$end, temp$row, lty=temp$lty, col=as.character(temp$col), lwd=temp$lwd, length = 0, angle = 90, code = 3)
	text(10^((log10(temp$end)+log10(temp$conc))/2), temp$row, temp$note, pos=1, cex=def.n.cex)
}

temp<-plotData[plotData$type=="p",]
if (dim(temp)[1]>0) {
	temp$col[is.na(temp$col)]<-def.p.col[1+((floor(temp$row[is.na(temp$col)])-1)%%length(def.p.col))]
	temp$bg[is.na(temp$bg)]<-def.bg[1]
	temp$pch[is.na(temp$pch)]<-def.pch[1]
	temp$cex[is.na(temp$cex)]<-def.cex[1]
	points(temp$conc,temp$row, pch=temp$pch, bg=as.character(temp$bg), col=as.character(temp$col), cex=temp$cex)
	text(temp$conc,temp$row,temp$note, pos=3, cex=def.n.cex)
}

axis(1, at=log10Tck('x','major'), tcl= -0.4, format(log10Tck('x','major'), scientific=FALSE, drop0trailing=TRUE), hadj=0.7) # bottom
axis(1, at=log10Tck('x','minor'), tcl= -0.2, labels=NA) # bottom

for (i in unique(plotLabels$font)) {
	temp<-plotLabels[plotLabels$font==i,]
#	axis(2,at=temp$row, labels=temp$label, font=i, las=1, padj=0.4, lwd=0)
	text(10^(par('usr')[1]), temp$row, labels=temp$label, xpd=TRUE, pos=2, font=i)	
}

if (savePlot) {
  dev.off()
}
